/**
 * @file        GenericQueue.h
 * @brief       Generic queue (according to std::queue).
 * @since       04.06.2020
 * @copyright   Copyright 2020 dSPACE GmbH. All rights reserved.
 */

#ifndef GENERICQUEUE_H
#define GENERICQUEUE_H

/*----------------------------------------------------------------------------*/
/* INCLUDES                                                                   */
/*----------------------------------------------------------------------------*/

#include "../../BusConfiguration.h"
#include "../Config.h"

#include <Std_Types.h>


#ifdef __cplusplus
extern "C" {
#endif


/*----------------------------------------------------------------------------*/
/* DEFINES                                                                    */
/*----------------------------------------------------------------------------*/

#define QUEUE_MODULE_ID               0xFFu
#define QUEUE_VENDOR_ID               0x23u

#define QUEUE_SID_CLEAR               0x01u
#define QUEUE_SID_DEINIT              0x02u
#define QUEUE_SID_FRONT               0x03u
#define QUEUE_SID_GET_AVAILABLE_SIZE  0x04u
#define QUEUE_SID_GET_SIZE            0x05u
#define QUEUE_SID_INIT                0x06u
#define QUEUE_SID_POP                 0x07u
#define QUEUE_SID_PULL                0x08u
#define QUEUE_SID_PUSH                0x09u

#define QUEUE_E_INSTANCE              0x01u
#define QUEUE_E_UNINIT                0x02u
#define QUEUE_E_PARAM                 0x03u


/*----------------------------------------------------------------------------*/
/* TYPEDEFS / ENUMS                                                           */
/*----------------------------------------------------------------------------*/

/**
 * @brief Describes a queue object.
 *
 * Do not modify content directly. Only to be used with methods of this module.
 *
 * Queue can store MaxSize-1 bytes.
 * If First == End, queue is empty.
 * If First/End reaches MaxSize, continue at 0 (First/End % MaxSize).
 */
typedef struct Queue_ObjectType {
        uint8  InstanceId;  /**< AUTOSAR instance ID. */
        uint32 First;       /**< First position (data is remove from here). */
        uint32 End;         /**< End position (data is inserted here). */
        uint32 MaxSize;     /**< Maximum size of queue (size of allocated memory). */
        uint8* Root;        /**< Queue root. */
} Queue_ObjectType;

/*----------------------------------------------------------------------------*/
/* FUNCTION DECLARATIONS                                                      */
/*----------------------------------------------------------------------------*/

/**
 * @brief Remove all elements from queue.
 * @param[in,out] queue Queue handle.
 */
void Queue_Clear(Queue_ObjectType* queue);

/**
 * @brief Free queue memory.
 * @param[in,out] queue Queue handle.
 */
void Queue_DeInit(Queue_ObjectType* queue);

/**
 * @brief Get data from queue without removing it.
 * @param[in]  queue   Queue handle.
 * @param[out] data    Container for data to be returned.
 * @param[in]  length  Length of data to be returned.
 * @return Length of returned data.
 */
uint32 Queue_Front(const Queue_ObjectType* queue, void* data, uint32 length);

/**
 * @brief Returns the currently available size of queue.
 * @param[in] queue  Queue handle.
 * @return Current available size of queue.
 */
uint32 Queue_GetAvailableSize(const Queue_ObjectType* queue);

/**
 * @brief Returns the current size (used memory) of queue.
 * @param[in] queue  Queue handle.
 * @return Current size of queue.
 */
uint32 Queue_GetSize(const Queue_ObjectType* queue);

/**
 * @brief Initialize queue, allocate memory.
 * @param[in,out] queue    Queue handle.
 * @param[in]     maxSize  Maximum size of queue (size of allocated memory, queue can only store maxSize-1).
 * @return E_OK on success, E_NOT_OK on failure.
 *
 * Since queue is defined to be empty, if First == End, the queue can only store maxSize-1 bytes.
 */
Std_ReturnType Queue_Init(Queue_ObjectType* queue, uint32 maxSize);

/**
 * @brief Remove data from queue.
 * @param[in,out] queue   Queue handle.
 * @param[in]     length  Length of data to be removed from queue.
 * @return Length of removed data.
 */
uint32 Queue_Pop(Queue_ObjectType* queue, uint32 length);

/**
 * @brief Take data from queue.
 * @param[in,out] queue   Queue handle.
 * @param[out]    data    Container for data to be returned (NULL_PTR will just delete data from queue).
 * @param[in]     length  Length of data to be returned.
 * @return Length of returned data.
 */
uint32 Queue_Pull(Queue_ObjectType* queue, void* data, uint32 length);

/**
 * @brief Add data to queue.
 * @param[in,out] queue   Queue handle.
 * @param[in]     data    Element to be added.
 * @param[in]     length  Length of data to be added.
 * @return E_OK on success, E_NOT_OK on error or if queue is full.
 */
Std_ReturnType Queue_Push(Queue_ObjectType* queue, const void* data, uint32 length);


#ifdef __cplusplus
}
#endif


#endif  /* GENERICQUEUE_H */

/*----------------------------------------------------------------------------*/
/* END OF FILE                                                                */
/*----------------------------------------------------------------------------*/
